<?php
/**
 * Content Generator
 * 
 * Uses OpenAI ChatGPT API to generate Google Ads compliant website content.
 * Follows strict guidelines for informational, editorial content.
 */

class ContentGenerator {
    private $apiKey;
    private $model = 'gpt-4o';
    private $apiUrl = 'https://api.openai.com/v1/chat/completions';
    
    public function __construct($apiKey) {
        $this->apiKey = $apiKey;
    }
    
    /**
     * Generate all content for the website
     */
    public function generate($domain, $businessType, $industry) {
        $businessName = $this->generateBusinessName($domain);
        
        // Build the comprehensive prompt
        $prompt = $this->buildPrompt($businessName, $businessType, $industry, $domain);
        
        // Call ChatGPT API
        $response = $this->callApi($prompt);
        
        if (!$response) {
            throw new Exception('Failed to get response from AI');
        }
        
        // Parse the JSON response
        $content = json_decode($response, true);
        
        if (!$content) {
            throw new Exception('Failed to parse AI response');
        }
        
        // Add metadata
        $content['domain'] = $domain;
        $content['business_name'] = $businessName;
        $content['business_type'] = $businessType;
        $content['industry'] = $industry;
        $content['generated_at'] = date('Y-m-d H:i:s');
        
        return $content;
    }
    
    /**
     * Generate a business name from domain
     */
    private function generateBusinessName($domain) {
        // Remove TLD
        $name = preg_replace('/\.[a-z]+$/', '', $domain);
        // Convert hyphens to spaces
        $name = str_replace(['-', '_'], ' ', $name);
        // Title case
        $name = ucwords($name);
        return $name;
    }
    
    /**
     * Build the comprehensive content generation prompt
     */
    private function buildPrompt($businessName, $businessType, $industry, $domain) {
        $businessTypeLabel = $this->getBusinessTypeLabel($businessType);
        
        return <<<PROMPT
You are a professional content writer creating website content for a {$businessTypeLabel} business.

BUSINESS DETAILS:
- Business Name: {$businessName}
- Industry/Niche: {$industry}
- Domain: {$domain}

CRITICAL CONTENT GUIDELINES (MUST FOLLOW):

1. INFORMATIONAL TONE ONLY
- Write as an editorial/informational resource
- Use neutral language: "explains", "covers", "introduces", "helps understand"
- NO promises, guarantees, or results claims
- NO medical, legal, or financial advice
- NO words like: "best", "top", "guaranteed", "proven", "cure", "fix"

2. CONTENT DEPTH
- Each section must have substantial content (80+ words minimum)
- Write in full paragraphs, not bullet points
- Explain WHAT, WHY, HOW for each topic
- Include context and background information

3. HUMAN-LIKE WRITING
- Vary sentence lengths and structures
- Use natural transitions between ideas
- Include specific details relevant to the industry
- Avoid repetitive patterns or AI-sounding phrases

4. CLEAR PURPOSE
- Every section should have a clear reason to exist
- Content should help users understand the topic
- Focus on education and information, not sales

Generate a complete JSON response with this exact structure:

{
    "meta": {
        "title": "Page title for SEO (50-60 chars)",
        "description": "Meta description for SEO (150-160 chars)"
    },
    "hero": {
        "headline": "Main headline (clear, not salesy)",
        "subheadline": "Supporting text explaining what visitors will find here (2-3 sentences)"
    },
    "about": {
        "title": "Section title",
        "content": "3-4 paragraphs about the business, its history, approach, and what makes it different. Focus on information, not promotion. Minimum 200 words."
    },
    "services": [
        {
            "title": "Service/offering name",
            "description": "Detailed explanation of this service, what it involves, who it's for, and what to expect. Minimum 80 words."
        }
    ],
    "process": {
        "title": "How It Works / Our Process",
        "intro": "Brief intro paragraph",
        "steps": [
            {
                "title": "Step title",
                "description": "Detailed explanation of this step"
            }
        ]
    },
    "faq": [
        {
            "question": "Common question",
            "answer": "Comprehensive answer with context and explanation. Minimum 60 words."
        }
    ],
    "contact": {
        "title": "Contact section title",
        "intro": "Welcoming text about getting in touch",
        "address": "Generate a realistic US address for the business type",
        "phone": "Generate a realistic US phone number",
        "email": "contact@{$domain}",
        "hours": "Realistic business hours"
    },
    "testimonials": [
        {
            "quote": "Realistic, specific testimonial focusing on experience not results",
            "author": "First name and last initial",
            "location": "City, State"
        }
    ],
    "footer": {
        "tagline": "Brief company description (1 sentence)",
        "copyright": "© 2025 {$businessName}. All rights reserved."
    },
    "privacy_policy": "Generate a complete, legitimate privacy policy for a {$businessTypeLabel}. Include sections on: Information Collection, How We Use Information, Cookies, Third-Party Services, Data Security, Your Rights, Contact Information, Policy Updates. Minimum 800 words.",
    "terms_of_service": "Generate complete terms of service. Include sections on: Acceptance of Terms, Description of Service, User Responsibilities, Intellectual Property, Disclaimers, Limitation of Liability, Governing Law, Contact Information. Minimum 800 words."
}

IMPORTANT:
- Generate exactly 4 services
- Generate exactly 6 FAQ items
- Generate exactly 3 testimonials
- All content must be original and specific to the industry
- Privacy Policy and Terms must be comprehensive legal documents
- Return ONLY valid JSON, no markdown or explanation

PROMPT;
    }
    
    /**
     * Get human-readable business type label
     */
    private function getBusinessTypeLabel($type) {
        $labels = [
            'repair_shop' => 'repair and service',
            'retail_store' => 'retail',
            'rental' => 'rental and accommodation',
            'fashion' => 'fashion and apparel',
            'wellness' => 'health and wellness',
            'beauty' => 'beauty and personal care',
            'fitness' => 'fitness and athletics',
            'medical' => 'healthcare',
            'restaurant' => 'food service',
            'travel' => 'travel and tourism',
            'event' => 'event planning',
            'photography' => 'creative services',
            'education' => 'education and training',
            'tech' => 'technology',
            'realestate' => 'real estate',
            'pet' => 'pet services',
            'professional' => 'professional services',
            'legal' => 'legal services',
            'finance' => 'financial services',
            'construction' => 'construction and contracting',
            'automotive' => 'automotive services',
            'cleaning' => 'cleaning services',
            'ecommerce' => 'e-commerce',
            'other' => 'general business'
        ];
        
        return $labels[$type] ?? 'business';
    }
    
    /**
     * Call OpenAI API
     */
    private function callApi($prompt) {
        $data = [
            'model' => $this->model,
            'messages' => [
                [
                    'role' => 'system',
                    'content' => 'You are a professional website content writer. You create informational, editorial content that is Google Ads compliant. You always respond with valid JSON only, no markdown formatting.'
                ],
                [
                    'role' => 'user',
                    'content' => $prompt
                ]
            ],
            'temperature' => 0.7,
            'max_tokens' => 8000,
            'response_format' => ['type' => 'json_object']
        ];
        
        $ch = curl_init($this->apiUrl);
        curl_setopt_array($ch, [
            CURLOPT_RETURNTRANSFER => true,
            CURLOPT_POST => true,
            CURLOPT_HTTPHEADER => [
                'Content-Type: application/json',
                'Authorization: Bearer ' . $this->apiKey
            ],
            CURLOPT_POSTFIELDS => json_encode($data),
            CURLOPT_TIMEOUT => 120
        ]);
        
        $response = curl_exec($ch);
        $httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
        $error = curl_error($ch);
        curl_close($ch);
        
        if ($error) {
            throw new Exception('API request failed: ' . $error);
        }
        
        if ($httpCode !== 200) {
            $errorData = json_decode($response, true);
            $errorMsg = $errorData['error']['message'] ?? 'Unknown API error';
            throw new Exception('API error (' . $httpCode . '): ' . $errorMsg);
        }
        
        $result = json_decode($response, true);
        
        if (!isset($result['choices'][0]['message']['content'])) {
            throw new Exception('Invalid API response structure');
        }
        
        return $result['choices'][0]['message']['content'];
    }
}
